const mongoose = require("mongoose");
const slugify = require("slugify");

const campusSchema = new mongoose.Schema(
  {
    name: {
      type: String,
      required: [true, "A campus must have a name"],
      trim: true,
      unique: true,
    },
    slug: String,
    subtitle: { type: String, trim: true },
    coverImage: {
      type: String,
      required: [true, "A campus must have a cover image"],
    },
    images: [String],
    promoVideo: String,
    description: { type: String, trim: true },
    floor: {
      type: String,
      required: [true, "A campus must specify a floor"],
      enum: ["Ground", "First", "Second", "Third", "Fourth"],
      unique: true,
    },
  },
  {
    timestamps: true,
    toJSON: { virtuals: true },
    toObject: { virtuals: true },
  }
);

// ✅ FIX: Point foreignField to 'campus._id' to match the embedded structure
campusSchema.virtual("courses", {
  ref: "Course",
  foreignField: "campus._id", // <--- CHANGED FROM 'campus' TO 'campus._id'
  localField: "_id",
});

// Slugify Middleware
campusSchema.pre("save", function (next) {
  if (this.name) {
    this.slug = slugify(this.name, { lower: true });
  }
  next();
});

const Campus = mongoose.model("Campus", campusSchema);
module.exports = Campus;
